<?php
session_start();
require_once '../config/database.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Non autorisé']);
    exit();
}

$current_user_id = $_SESSION['user_id'];
$search_query = isset($_GET['q']) ? trim($_GET['q']) : '';

if (empty($search_query) || strlen($search_query) < 2) {
    echo json_encode(['success' => true, 'data' => []]); // Return empty if query is too short
    exit();
}

$database = new Database();
$db = $database->getConnection();

try {
    // Search for users by display_name or email, excluding the current user
    $query = "SELECT id, display_name, avatar_url FROM users 
              WHERE (display_name LIKE :search_query OR email LIKE :search_query)
              AND id != :current_user_id
              LIMIT 10";
    
    $stmt = $db->prepare($query);
    $search_param = "%{$search_query}%";
    $stmt->bindParam(':search_query', $search_param);
    $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
    $stmt->execute();
    
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // For each found user, check friendship status with the current user
    $results = [];
    foreach ($users as $user) {
        $user_one_id = min($current_user_id, $user['id']);
        $user_two_id = max($current_user_id, $user['id']);

        $friend_status_query = "SELECT status, action_user_id FROM friends 
                                WHERE user_one_id = :user_one_id AND user_two_id = :user_two_id";
        $friend_stmt = $db->prepare($friend_status_query);
        $friend_stmt->bindParam(':user_one_id', $user_one_id, PDO::PARAM_INT);
        $friend_stmt->bindParam(':user_two_id', $user_two_id, PDO::PARAM_INT);
        $friend_stmt->execute();
        $friendship = $friend_stmt->fetch(PDO::FETCH_ASSOC);

        $user['friend_status'] = null; // Not friends, no pending request
        if ($friendship) {
            if ($friendship['status'] === 'accepted') {
                $user['friend_status'] = 'friends';
            } elseif ($friendship['status'] === 'pending') {
                if ($friendship['action_user_id'] == $current_user_id) {
                    $user['friend_status'] = 'request_sent'; // Current user sent request to this user
                } else {
                    $user['friend_status'] = 'request_received'; // Current user received request from this user
                }
            }
        }
        $results[] = $user;
    }
    
    echo json_encode(['success' => true, 'data' => $results]);

} catch (PDOException $e) {
    http_response_code(500);
    error_log("User Search API Error: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Erreur serveur.', 'details' => $e->getMessage()]);
}
?>
